<?php
session_start();
require_once '../config/db.php';
require_once '../helpers/helpers.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// Check if user has permission to edit invoices
if (!hasPermission('edit_invoice')) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Permission denied']);
    exit;
}

try {
    // Enable error reporting for debugging
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    
    // Log the raw input
    $raw_input = file_get_contents('php://input');
    error_log("Raw input received: " . $raw_input);
    
    $input = json_decode($raw_input, true);
    
    if (!$input) {
        $json_error = json_last_error_msg();
        error_log("JSON decode error: " . $json_error);
        throw new Exception('Invalid input data: ' . $json_error);
    }
    
    error_log("Parsed input: " . print_r($input, true));
    
    $invoice_id = $input['invoice_id'] ?? null;
    $customer_id = $input['customer_id'] ?? null;
    $issue_date = $input['issue_date'] ?? null;
    $due_date = $input['due_date'] ?? null;
    $notes = $input['notes'] ?? '';
    $items = $input['items'] ?? [];
    
    if (!$invoice_id) {
        throw new Exception('Invoice ID is required');
    }
    
    if (!$customer_id) {
        throw new Exception('Customer is required');
    }
    
    if (!$issue_date) {
        throw new Exception('Issue date is required');
    }
    
    if (!$due_date) {
        throw new Exception('Due date is required');
    }
    
    if (empty($items)) {
        throw new Exception('At least one item is required');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    // Check if invoice exists and is not paid
    $stmt = $pdo->prepare("SELECT status FROM invoices WHERE id = ?");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$invoice) {
        throw new Exception('Invoice not found');
    }
    
    if ($invoice['status'] === 'paid') {
        throw new Exception('Cannot edit paid invoices');
    }
    
    // Calculate totals
    $subtotal = 0;
    $tax_amount = 0;
    $total_amount = 0;
    
    foreach ($items as $item) {
        $quantity = floatval($item['quantity']);
        $unit_price = floatval($item['unit_price']);
        $total_price = $quantity * $unit_price;
        $subtotal += $total_price;
    }
    
    // Use fixed 0% tax rate (same as create_invoice.php)
    $tax_rate = 0.00;
    $tax_amount = $subtotal * $tax_rate;
    $total_amount = $subtotal + $tax_amount;
    
    // Update invoice
    $stmt = $pdo->prepare("
        UPDATE invoices 
        SET customer_id = ?, issue_date = ?, due_date = ?, notes = ?, 
            subtotal = ?, tax_amount = ?, total_amount = ?, updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([
        $customer_id, $issue_date, $due_date, $notes,
        $subtotal, $tax_amount, $total_amount, $invoice_id
    ]);
    
    // Delete existing invoice items
    $stmt = $pdo->prepare("DELETE FROM invoice_items WHERE invoice_id = ?");
    $stmt->execute([$invoice_id]);
    
    // Insert new invoice items
    $stmt = $pdo->prepare("
        INSERT INTO invoice_items (invoice_id, product_id, quantity, unit_price, total_price)
        VALUES (?, ?, ?, ?, ?)
    ");
    
    foreach ($items as $item) {
        $product_id = $item['product_id'] ?? null;
        $quantity = floatval($item['quantity']);
        $unit_price = floatval($item['unit_price']);
        $total_price = $quantity * $unit_price;
        
        $stmt->execute([
            $invoice_id, $product_id, $quantity, $unit_price, $total_price
        ]);
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Invoice updated successfully',
        'invoice_id' => $invoice_id,
        'total_amount' => $total_amount
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    // Log the error for debugging
    error_log("Invoice update error: " . $e->getMessage());
    error_log("File: " . $e->getFile() . " Line: " . $e->getLine());
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]
    ]);
}
?> 